/* SPDX-License-Identifier: GPL-2.0-only */

#include <bootstate.h>
#include <cpu/x86/msr.h>
#include <device/device.h>
#include <drivers/intel/gma/int15.h>
#include <gpio.h>
#include <option.h>
#include "sch5555_ec.h"

static void mainboard_enable(struct device *dev)
{
	install_intel_vga_int15_handler(GMA_INT15_ACTIVE_LFP_NONE,
					GMA_INT15_PANEL_FIT_DEFAULT,
					GMA_INT15_BOOT_DISPLAY_DEFAULT, 0);
}

struct chip_operations mainboard_ops = {
	.enable_dev = mainboard_enable,
};

#define HWM_TAB_ADD_TEMP_TARGET	1
#define HWM_TAB_PKG_POWER_ANY	0xffff
#define CHASSIS_TYPE_UNKNOWN	0xff

struct hwm_tab_entry {
	uint16_t addr;
	uint8_t val;
	uint8_t flags;
	uint16_t pkg_power;
};

struct hwm_tab_entry HWM_TAB3[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x00, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0x8a, 0, 0x0010 },
	{ 0x086, 0x4c, 0, 0x0010 },
	{ 0x08a, 0x66, 0, 0x0010 },
	{ 0x08b, 0x5b, 0, 0x0010 },
	{ 0x090, 0x65, 0, 0xffff },
	{ 0x091, 0x70, 0, 0xffff },
	{ 0x092, 0x86, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0xa4, 0, 0xffff },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x0e, 0, 0xffff },
	{ 0x0a1, 0x0e, 0, 0xffff },
	{ 0x0ae, 0x7c, 0, 0xffff },
	{ 0x0af, 0x86, 0, 0xffff },
	{ 0x0b0, 0x9a, 0, 0xffff },
	{ 0x0b3, 0x9a, 0, 0xffff },
	{ 0x0b6, 0x08, 0, 0xffff },
	{ 0x0b7, 0x08, 0, 0xffff },
	{ 0x0ea, 0x64, 0, 0x0020 },
	{ 0x0ea, 0x5c, 0, 0x0010 },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x0fd, 0x01, 0, 0xffff },
	{ 0x1a1, 0x00, 0, 0xffff },
	{ 0x1a2, 0x00, 0, 0xffff },
	{ 0x1b1, 0x08, 0, 0xffff },
	{ 0x1be, 0x99, 0, 0xffff },
	{ 0x280, 0xa0, 0, 0x0010 },
	{ 0x281, 0x0f, 0, 0x0010 },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff },
	{ 0x288, 0x68, 0, 0x0010 },
	{ 0x289, 0x10, 0, 0x0010 },
	{ 0x28a, 0x03, 0, 0xffff },
	{ 0x28b, 0x0a, 0, 0xffff },
	{ 0x28c, 0x80, 0, 0xffff },
	{ 0x28d, 0x03, 0, 0xffff },
};

struct hwm_tab_entry HWM_TAB4[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x00, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0x99, 0, 0x0020 },
	{ 0x085, 0xad, 0, 0x0010 },
	{ 0x086, 0x1c, 0, 0xffff },
	{ 0x08a, 0x39, 0, 0x0020 },
	{ 0x08a, 0x41, 0, 0x0010 },
	{ 0x08b, 0x76, 0, 0x0020 },
	{ 0x08b, 0x8b, 0, 0x0010 },
	{ 0x090, 0x5e, 0, 0xffff },
	{ 0x091, 0x5e, 0, 0xffff },
	{ 0x092, 0x86, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0xa4, 0, 0xffff },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x0a, 0, 0xffff },
	{ 0x0a1, 0x0a, 0, 0xffff },
	{ 0x0ae, 0x7c, 0, 0xffff },
	{ 0x0af, 0x7c, 0, 0xffff },
	{ 0x0b0, 0x9a, 0, 0xffff },
	{ 0x0b3, 0x7c, 0, 0xffff },
	{ 0x0b6, 0x08, 0, 0xffff },
	{ 0x0b7, 0x08, 0, 0xffff },
	{ 0x0ea, 0x64, 0, 0x0020 },
	{ 0x0ea, 0x5c, 0, 0x0010 },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x0fd, 0x01, 0, 0xffff },
	{ 0x1a1, 0x00, 0, 0xffff },
	{ 0x1a2, 0x00, 0, 0xffff },
	{ 0x1b1, 0x08, 0, 0xffff },
	{ 0x1be, 0x90, 0, 0xffff },
	{ 0x280, 0x94, 0, 0x0020 },
	{ 0x281, 0x11, 0, 0x0020 },
	{ 0x280, 0x94, 0, 0x0010 },
	{ 0x281, 0x11, 0, 0x0010 },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff },
	{ 0x288, 0x28, 0, 0x0020 },
	{ 0x289, 0x0a, 0, 0x0020 },
	{ 0x288, 0x28, 0, 0x0010 },
	{ 0x289, 0x0a, 0, 0x0010 },
	{ 0x28a, 0x03, 0, 0xffff },
	{ 0x28b, 0x0a, 0, 0xffff },
	{ 0x28c, 0x80, 0, 0xffff },
	{ 0x28d, 0x03, 0, 0xffff },
};

struct hwm_tab_entry HWM_TAB5[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x00, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0x66, 0, 0x0020 },
	{ 0x085, 0x5d, 0, 0x0010 },
	{ 0x086, 0x1c, 0, 0xffff },
	{ 0x08a, 0x39, 0, 0x0020 },
	{ 0x08a, 0x41, 0, 0x0010 },
	{ 0x08b, 0x76, 0, 0x0020 },
	{ 0x08b, 0x80, 0, 0x0010 },
	{ 0x090, 0x5d, 0, 0x0020 },
	{ 0x090, 0x5e, 0, 0x0010 },
	{ 0x091, 0x5e, 0, 0xffff },
	{ 0x092, 0x86, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0xa3, 0, 0x0020 },
	{ 0x098, 0xa4, 0, 0x0010 },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x08, 0, 0xffff },
	{ 0x0a1, 0x0a, 0, 0xffff },
	{ 0x0ae, 0x7c, 0, 0xffff },
	{ 0x0af, 0x7c, 0, 0xffff },
	{ 0x0b0, 0x9a, 0, 0xffff },
	{ 0x0b3, 0x7c, 0, 0xffff },
	{ 0x0b6, 0x08, 0, 0xffff },
	{ 0x0b7, 0x08, 0, 0xffff },
	{ 0x0ea, 0x64, 0, 0x0020 },
	{ 0x0ea, 0x5c, 0, 0x0010 },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x0fd, 0x01, 0, 0xffff },
	{ 0x1a1, 0x00, 0, 0xffff },
	{ 0x1a2, 0x00, 0, 0xffff },
	{ 0x1b1, 0x08, 0, 0xffff },
	{ 0x1be, 0x98, 0, 0x0020 },
	{ 0x1be, 0x90, 0, 0x0010 },
	{ 0x280, 0x94, 0, 0x0020 },
	{ 0x281, 0x11, 0, 0x0020 },
	{ 0x280, 0x94, 0, 0x0010 },
	{ 0x281, 0x11, 0, 0x0010 },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff },
	{ 0x288, 0x28, 0, 0x0020 },
	{ 0x289, 0x0a, 0, 0x0020 },
	{ 0x288, 0x28, 0, 0x0010 },
	{ 0x289, 0x0a, 0, 0x0010 },
	{ 0x28a, 0x03, 0, 0xffff },
	{ 0x28b, 0x0a, 0, 0xffff },
	{ 0x28c, 0x80, 0, 0xffff },
	{ 0x28d, 0x03, 0, 0xffff },
};

struct hwm_tab_entry HWM_TAB6[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x00, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0x98, 0, 0xffff },
	{ 0x086, 0x3c, 0, 0xffff },
	{ 0x08a, 0x39, 0, 0x0020 },
	{ 0x08a, 0x3d, 0, 0x0010 },
	{ 0x08b, 0x44, 0, 0x0020 },
	{ 0x08b, 0x51, 0, 0x0010 },
	{ 0x090, 0x61, 0, 0xffff },
	{ 0x091, 0x6d, 0, 0xffff },
	{ 0x092, 0x86, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0x9f, 0, 0x0020 },
	{ 0x098, 0xa4, 0, 0x0010 },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x0e, 0, 0xffff },
	{ 0x0a1, 0x0e, 0, 0xffff },
	{ 0x0ae, 0x7c, 0, 0xffff },
	{ 0x0af, 0x7c, 0, 0xffff },
	{ 0x0b0, 0x9b, 0, 0x0020 },
	{ 0x0b0, 0x98, 0, 0x0010 },
	{ 0x0b3, 0x9a, 0, 0xffff },
	{ 0x0b6, 0x08, 0, 0xffff },
	{ 0x0b7, 0x08, 0, 0xffff },
	{ 0x0ea, 0x64, 0, 0x0020 },
	{ 0x0ea, 0x5c, 0, 0x0010 },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x0fd, 0x01, 0, 0xffff },
	{ 0x1a1, 0x00, 0, 0xffff },
	{ 0x1a2, 0x00, 0, 0xffff },
	{ 0x1b1, 0x08, 0, 0xffff },
	{ 0x1be, 0x9a, 0, 0x0020 },
	{ 0x1be, 0x96, 0, 0x0010 },
	{ 0x280, 0x94, 0, 0x0020 },
	{ 0x281, 0x11, 0, 0x0020 },
	{ 0x280, 0x94, 0, 0x0010 },
	{ 0x281, 0x11, 0, 0x0010 },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff },
	{ 0x288, 0x94, 0, 0x0020 },
	{ 0x289, 0x11, 0, 0x0020 },
	{ 0x288, 0x94, 0, 0x0010 },
	{ 0x289, 0x11, 0, 0x0010 },
	{ 0x28a, 0x03, 0, 0xffff },
	{ 0x28b, 0x0a, 0, 0xffff },
	{ 0x28c, 0x80, 0, 0xffff },
	{ 0x28d, 0x03, 0, 0xffff },
};

static uint8_t get_chassis_type(void)
{
	uint8_t gpio_chassis_type;

	// Read chassis type from GPIO
	const gpio_t chassis_id_pins[] = {1, 17, 38, 70};
	gpio_chassis_type = gpio_base2_value(chassis_id_pins, ARRAY_SIZE(chassis_id_pins));

	printk(BIOS_DEBUG, "GPIO chassis type = %#x\n", gpio_chassis_type);

	// Turn it into internal chassis index
	switch (gpio_chassis_type) {
	case 0x08:
	case 0x0a:
		return 4;
	case 0x0b:
		return 3;
	case 0x0c:
		return 5;
	case 0x0d: // SFF
	case 0x0e:
	case 0x0f:
		return 6;
	default:
		return CHASSIS_TYPE_UNKNOWN;
	}

}

static uint8_t get_temp_target(void)
{
	uint8_t val = rdmsr(0x1a2).lo >> 8 & 0xff;
	if (!val)
		val = 20;
	return 0x95 - val;
}

static uint16_t get_pkg_power(void)
{
	const unsigned int pkg_power = rdmsr(0x614).lo & 0x7fff;
	const unsigned int power_unit = 1 << (rdmsr(0x606).lo & 0xf);
	if (pkg_power / power_unit > 65)
		return 32;
	else
		return 16;
}

static void apply_hwm_tab(struct hwm_tab_entry *arr, size_t size)
{
	uint8_t temp_target = get_temp_target();
	uint16_t pkg_power = get_pkg_power();

	printk(BIOS_DEBUG, "Temp target = %#x\n", temp_target);
	printk(BIOS_DEBUG, "Package power = %#x\n", pkg_power);

	for (size_t i = 0; i < size; ++i) {
		// Skip entry if it doesn't apply for this package power
		if (arr[i].pkg_power != pkg_power &&
		    arr[i].pkg_power != HWM_TAB_PKG_POWER_ANY)
			continue;

		uint8_t val = arr[i].val;

		// Add temp target to value if requested (current tables never do)
		if (arr[i].flags & HWM_TAB_ADD_TEMP_TARGET)
			val += temp_target;

		// Perform write
		sch5555_mbox_write(1, arr[i].addr, val);

	}
}

static void sch5555_ec_hwm_init(void *arg)
{
	uint8_t chassis_type, saved_2fc;

	printk(BIOS_DEBUG, "OptiPlex 9020 late HWM init\n");

	saved_2fc = sch5555_mbox_read(1, 0x2fc);
	sch5555_mbox_write(1, 0x2fc, 0xa0);
	sch5555_mbox_write(1, 0x2fd, 0x32);

	chassis_type = get_chassis_type();

	if (chassis_type != CHASSIS_TYPE_UNKNOWN) {
		printk(BIOS_DEBUG, "Chassis type = %#x\n", chassis_type);
	} else {
		printk(BIOS_DEBUG, "WARNING: Unknown chassis type\n");
	}

	// Apply HWM table based on chassis type
	switch (chassis_type) {
	case 3:
		apply_hwm_tab(HWM_TAB3, ARRAY_SIZE(HWM_TAB3));
		break;
	case 4:
		apply_hwm_tab(HWM_TAB4, ARRAY_SIZE(HWM_TAB4));
		break;
	case 5:
		apply_hwm_tab(HWM_TAB5, ARRAY_SIZE(HWM_TAB5));
		break;
	case 6:
		apply_hwm_tab(HWM_TAB6, ARRAY_SIZE(HWM_TAB6));
		break;
	}

	// NOTE: vendor firmware applies these when "max core address" > 2
	// i think this is always the case
	sch5555_mbox_write(1, 0x9e, 0x30);
	sch5555_mbox_write(1, 0xeb, sch5555_mbox_read(1, 0xea));

	sch5555_mbox_write(1, 0x2fc, saved_2fc);

	// Apply full speed fan config if requested or if the chassis type is unknown
	if (chassis_type == CHASSIS_TYPE_UNKNOWN || get_uint_option("fan_full_speed", 0)) {
		printk(BIOS_DEBUG, "Setting full fan speed\n");
		sch5555_mbox_write(1, 0x80, 0x60 | sch5555_mbox_read(1, 0x80));
		sch5555_mbox_write(1, 0x81, 0x60 | sch5555_mbox_read(1, 0x81));
	}

	sch5555_mbox_read(1, 0xb8);

	if ((chassis_type == 4 || chassis_type == 5) && sch5555_mbox_read(1, 0x26) == 0) {
		sch5555_mbox_write(1, 0xa0, sch5555_mbox_read(1, 0xa0) & 0xfb);
		sch5555_mbox_write(1, 0xa1, sch5555_mbox_read(1, 0xa1) & 0xfb);
		sch5555_mbox_write(1, 0xa2, sch5555_mbox_read(1, 0xa2) & 0xfb);
		sch5555_mbox_write(1, 0x8a, 0x99);
		sch5555_mbox_write(1, 0x8b, 0x47);
		sch5555_mbox_write(1, 0x8c, 0x91);
	}
}

BOOT_STATE_INIT_ENTRY(BS_POST_DEVICE, BS_ON_EXIT, sch5555_ec_hwm_init, NULL);
