import pytest
from helpers import parse_inchi_from_executable_output


@pytest.fixture
def molfile_CID_166625356():
    return """
  -OEChem-10082407492D

 90 99  0     1  0  0  0  0  0999 V2000
   12.2611    4.7992    0.0000 S   0  0  0  0  0  0  0  0  0  0  0  0
    9.3177   -3.9691    0.0000 S   0  0  0  0  0  0  0  0  0  0  0  0
    3.5049    2.9843    0.0000 S   0  0  0  0  0  0  0  0  0  0  0  0
    9.4120    1.3332    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
   14.6081    1.3332    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
   12.8761    1.3332    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
    9.2780    1.8332    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    8.3959    0.2917    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    7.5459    1.8332    0.0000 N   0  3  0  0  0  0  0  0  0  0  0  0
   10.1960    0.2847    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    8.4120    3.3332    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    6.6730    0.3468    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
   14.6081   -0.6668    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    9.2780    2.8332    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    9.2939   -0.2361    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.6865    1.3384    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.1880    1.3263    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.5188   -0.1711    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.5459    2.8332    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.2610    2.9791    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.8935   -1.2033    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.0666    2.1158    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.7748    2.1284    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.8474   -1.1650    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.5622    2.9777    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.7468    3.8932    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    9.4186   -2.1590    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    5.0312    2.0980    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   11.8100    2.1328    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.2538   -2.0798    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.0567    3.8816    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.2780    0.8333    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   11.7816    3.9216    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.8464   -3.0872    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    4.5048    2.9899    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   12.3168    3.0355    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.7567   -3.0473    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    5.0211    3.8878    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.2780   -0.1668    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   11.1440    1.3332    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   11.1441   -0.6668    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   12.0100    0.8332    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   12.0101   -0.1667    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   12.8761   -0.6667    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   13.2608    4.8227    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.7896   -4.8183    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    3.0000    3.8475    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   13.7421   -0.1668    0.0000 C   0  0  1  0  0  0  0  0  0  0  0  0
   13.7403    5.7003    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    9.2610   -5.7003    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    2.0000    3.8419    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   13.7421    0.8332    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   14.6081    2.3332    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.4120    1.3332    0.0000 B   0  5  2  0  0  0  0  0  0  0  0  0
   10.4202    4.4202    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   10.0384   -2.1745    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    4.7284    1.5570    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   12.1243    1.5984    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.6345   -2.0499    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.3719    4.4155    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   12.9367    3.0453    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    7.4252   -3.5713    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    4.7122    4.4253    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.7411   -0.4768    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   11.1440    1.9532    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   11.1441   -1.2868    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   12.5470    1.1432    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   12.4776   -1.1417    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.2746   -1.1417    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.1676    4.2098    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.8483    4.6244    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.3280   -4.4045    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.3019   -5.2011    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.5814    4.0628    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.8889    4.4575    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.7421   -0.7868    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   14.2844    5.4030    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   14.0376    6.2444    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.1962    5.9975    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.7142   -5.9925    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.5533   -6.2471    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.8078   -5.4080    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    1.9965    4.4618    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    1.3800    3.8384    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.0035    3.2219    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   15.1451   -0.3568    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   14.6081   -1.2868    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   15.2281    2.3332    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   14.6081    2.9532    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.9881    2.3332    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
  1 33  1  0  0  0  0
  1 45  1  0  0  0  0
  2 34  1  0  0  0  0
  2 46  1  0  0  0  0
  3 35  1  0  0  0  0
  3 47  1  0  0  0  0
  4 32  1  0  0  0  0
 54  4  1  6  0  0  0
  5 52  1  0  0  0  0
  5 53  1  0  0  0  0
  6 52  2  0  0  0  0
  7 14  1  0  0  0  0
  7 17  1  0  0  0  0
  7 54  1  0  0  0  0
  8 15  1  0  0  0  0
  8 18  1  0  0  0  0
  8 54  1  0  0  0  0
  9 16  1  0  0  0  0
  9 19  2  0  0  0  0
  9 54  1  0  0  0  0
 10 15  1  0  0  0  0
 10 17  2  0  0  0  0
 11 14  2  0  0  0  0
 11 19  1  0  0  0  0
 12 16  2  0  0  0  0
 12 18  1  0  0  0  0
 48 13  1  1  0  0  0
 13 86  1  0  0  0  0
 13 87  1  0  0  0  0
 14 20  1  0  0  0  0
 15 21  2  0  0  0  0
 16 22  1  0  0  0  0
 17 23  1  0  0  0  0
 18 24  2  0  0  0  0
 19 25  1  0  0  0  0
 20 23  2  0  0  0  0
 20 26  1  0  0  0  0
 21 24  1  0  0  0  0
 21 27  1  0  0  0  0
 22 25  2  0  0  0  0
 22 28  1  0  0  0  0
 23 29  1  0  0  0  0
 24 30  1  0  0  0  0
 25 31  1  0  0  0  0
 26 33  2  0  0  0  0
 26 55  1  0  0  0  0
 27 34  2  0  0  0  0
 27 56  1  0  0  0  0
 28 35  2  0  0  0  0
 28 57  1  0  0  0  0
 29 36  2  0  0  0  0
 29 58  1  0  0  0  0
 30 37  2  0  0  0  0
 30 59  1  0  0  0  0
 31 38  2  0  0  0  0
 31 60  1  0  0  0  0
 32 39  2  0  0  0  0
 32 40  1  0  0  0  0
 33 36  1  0  0  0  0
 34 37  1  0  0  0  0
 35 38  1  0  0  0  0
 36 61  1  0  0  0  0
 37 62  1  0  0  0  0
 38 63  1  0  0  0  0
 39 41  1  0  0  0  0
 39 64  1  0  0  0  0
 40 42  2  0  0  0  0
 40 65  1  0  0  0  0
 41 43  2  0  0  0  0
 41 66  1  0  0  0  0
 42 43  1  0  0  0  0
 42 67  1  0  0  0  0
 43 44  1  0  0  0  0
 44 48  1  0  0  0  0
 44 68  1  0  0  0  0
 44 69  1  0  0  0  0
 45 49  1  0  0  0  0
 45 70  1  0  0  0  0
 45 71  1  0  0  0  0
 46 50  1  0  0  0  0
 46 72  1  0  0  0  0
 46 73  1  0  0  0  0
 47 51  1  0  0  0  0
 47 74  1  0  0  0  0
 47 75  1  0  0  0  0
 48 52  1  0  0  0  0
 48 76  1  0  0  0  0
 49 77  1  0  0  0  0
 49 78  1  0  0  0  0
 49 79  1  0  0  0  0
 50 80  1  0  0  0  0
 50 81  1  0  0  0  0
 50 82  1  0  0  0  0
 51 83  1  0  0  0  0
 51 84  1  0  0  0  0
 51 85  1  0  0  0  0
 53 88  1  0  0  0  0
 53 89  1  0  0  0  0
 53 90  1  0  0  0  0
M  CHG  2   9   1  54  -1
M  END
"""


@pytest.fixture
def molfile_CID_6419481():
    return """
  -OEChem-10082407562D

 41 42  0     0  0  0  0  0  0999 V2000
    2.0000    2.8450    0.0000 S   0  5  0  0  0  0  0  0  0  0  0  0
    2.8660   -1.6550    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
    2.7320   -0.1550    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
    6.3301   -1.6550    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    6.3301   -2.6550    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    3.7320    1.8450    0.0000 N   0  3  0  0  0  0  0  0  0  0  0  0
    3.7320    2.8450    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660    4.3450    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    5.4641   -1.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981    0.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    5.4641   -0.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981   -1.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    3.7320   -1.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    3.7320   -0.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.1962   -3.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981    1.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.1962   -4.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -2.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -4.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.9282   -3.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.9282   -4.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660   -2.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660    1.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660    3.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.0010    0.1550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981   -2.2750    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.8671   -1.3450    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    5.7932   -2.9650    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    5.1350    1.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.6592   -4.4650    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -2.0350    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -5.2750    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.4651   -2.8450    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.4651   -4.4650    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.2460   -2.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660   -3.2750    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.4860   -2.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.3291    1.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.5561    0.8080    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.3291    4.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.4030    4.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
  1 24  1  0  0  0  0
  2 13  1  0  0  0  0
  2 22  1  0  0  0  0
  3 14  2  0  0  0  0
  4  5  1  0  0  0  0
  4  9  1  0  0  0  0
  4 27  1  0  0  0  0
  5 15  1  0  0  0  0
  5 28  1  0  0  0  0
  6  7  1  0  0  0  0
  6 16  1  0  0  0  0
  6 23  2  0  0  0  0
  7 24  2  0  0  0  0
  8 24  1  0  0  0  0
  8 40  1  0  0  0  0
  8 41  1  0  0  0  0
  9 11  2  0  0  0  0
  9 12  1  0  0  0  0
 10 11  1  0  0  0  0
 10 14  1  0  0  0  0
 10 16  2  0  0  0  0
 11 25  1  0  0  0  0
 12 13  2  0  0  0  0
 12 26  1  0  0  0  0
 13 14  1  0  0  0  0
 15 17  2  0  0  0  0
 15 18  1  0  0  0  0
 16 29  1  0  0  0  0
 17 19  1  0  0  0  0
 17 30  1  0  0  0  0
 18 20  2  0  0  0  0
 18 31  1  0  0  0  0
 19 21  2  0  0  0  0
 19 32  1  0  0  0  0
 20 21  1  0  0  0  0
 20 33  1  0  0  0  0
 21 34  1  0  0  0  0
 22 35  1  0  0  0  0
 22 36  1  0  0  0  0
 22 37  1  0  0  0  0
 23 38  1  0  0  0  0
 23 39  1  0  0  0  0
M  CHG  2   1  -1   6   1
M  END
"""


@pytest.fixture
def molfile_CID_6419538():
    return """6419481
  -OEChem-10082407562D

 41 42  0     0  0  0  0  0  0999 V2000
    2.0000    2.8450    0.0000 S   0  5  0  0  0  0  0  0  0  0  0  0
    2.8660   -1.6550    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
    2.7320   -0.1550    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
    6.3301   -1.6550    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    6.3301   -2.6550    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    3.7320    1.8450    0.0000 N   0  3  0  0  0  0  0  0  0  0  0  0
    3.7320    2.8450    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660    4.3450    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    5.4641   -1.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981    0.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    5.4641   -0.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981   -1.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    3.7320   -1.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    3.7320   -0.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.1962   -3.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981    1.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.1962   -4.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -2.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -4.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.9282   -3.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.9282   -4.1550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660   -2.6550    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660    1.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660    3.3450    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.0010    0.1550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    4.5981   -2.2750    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.8671   -1.3450    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    5.7932   -2.9650    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    5.1350    1.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.6592   -4.4650    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -2.0350    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.0622   -5.2750    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.4651   -2.8450    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.4651   -4.4650    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.2460   -2.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.8660   -3.2750    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.4860   -2.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.3291    1.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.5561    0.8080    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.3291    4.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.4030    4.6550    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
  1 24  1  0  0  0  0
  2 13  1  0  0  0  0
  2 22  1  0  0  0  0
  3 14  2  0  0  0  0
  4  5  1  0  0  0  0
  4  9  1  0  0  0  0
  4 27  1  0  0  0  0
  5 15  1  0  0  0  0
  5 28  1  0  0  0  0
  6  7  1  0  0  0  0
  6 16  1  0  0  0  0
  6 23  2  0  0  0  0
  7 24  2  0  0  0  0
  8 24  1  0  0  0  0
  8 40  1  0  0  0  0
  8 41  1  0  0  0  0
  9 11  2  0  0  0  0
  9 12  1  0  0  0  0
 10 11  1  0  0  0  0
 10 14  1  0  0  0  0
 10 16  2  0  0  0  0
 11 25  1  0  0  0  0
 12 13  2  0  0  0  0
 12 26  1  0  0  0  0
 13 14  1  0  0  0  0
 15 17  2  0  0  0  0
 15 18  1  0  0  0  0
 16 29  1  0  0  0  0
 17 19  1  0  0  0  0
 17 30  1  0  0  0  0
 18 20  2  0  0  0  0
 18 31  1  0  0  0  0
 19 21  2  0  0  0  0
 19 32  1  0  0  0  0
 20 21  1  0  0  0  0
 20 33  1  0  0  0  0
 21 34  1  0  0  0  0
 22 35  1  0  0  0  0
 22 36  1  0  0  0  0
 22 37  1  0  0  0  0
 23 38  1  0  0  0  0
 23 39  1  0  0  0  0
M  CHG  2   1  -1   6   1
M  END
"""


@pytest.fixture
def molfile_CID_53234134():
    return """53234134
  -OEChem-10152410132D

 75 78  0     1  0  0  0  0  0999 V2000
   10.2490    0.8123    0.0000 S   0  0  0  0  0  0  0  0  0  0  0  0
   16.8263    1.2831    0.0000 S   0  0  0  0  0  0  0  0  0  0  0  0
   12.8289    3.0250    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
    9.3829    4.3123    0.0000 O   0  5  0  0  0  0  0  0  0  0  0  0
   11.1150    4.3123    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
   14.0562    1.8218    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
   13.2738   -1.0744    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
    4.5792    0.9008    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
   13.4578   -3.7137    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
   11.7857   -3.2620    0.0000 O   0  0  0  0  0  0  0  0  0  0  0  0
   11.1150    2.3123    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
   12.8289    0.5996    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    7.6509    2.3123    0.0000 N   0  3  0  0  0  0  0  0  0  0  0  0
    6.7373    2.7190    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    6.1615    0.1963    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    5.0737    2.0804    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
   14.2402   -0.8176    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
   16.2433   -0.2262    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
   15.8278    1.3375    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    4.7602   -0.8218    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
   18.0160   -0.0436    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
    2.3644   -1.9444    0.0000 N   0  0  0  0  0  0  0  0  0  0  0  0
   11.1150    1.3123    0.0000 C   0  0  1  0  0  0  0  0  0  0  0  0
   12.1233    1.3081    0.0000 C   0  0  1  0  0  0  0  0  0  0  0  0
   12.1233    2.3164    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.2490    2.8123    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    9.3829    2.3123    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    9.3829    1.3123    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    8.5169    2.8123    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   10.2490    3.8123    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   13.7954    0.8564    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    7.5464    1.3178    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.0682    1.9759    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.5682    1.1098    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   14.5010    0.1478    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    6.5294    3.6972    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   15.4675    0.4047    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    5.1669    0.0918    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   13.0130   -2.0398    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   17.0831    0.3167    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   13.9783   -2.3006    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   12.0476   -1.7790    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    3.7657   -0.9263    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   12.7522   -3.0052    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
    3.3590   -1.8399    0.0000 C   0  0  0  0  0  0  0  0  0  0  0  0
   11.3333    0.4908    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   11.8842    0.7361    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.7724    1.4199    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    9.1709    0.7297    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   12.6672    0.0010    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.9154    3.2872    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.1184    3.2872    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    8.0071    0.9029    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    7.1359    3.8261    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.4005    4.3036    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    5.9230    3.5682    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    6.5259   -0.3053    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    4.8215    2.6468    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    4.7092    1.5788    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    5.1246   -1.3234    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.8167   -2.8991    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   14.5769   -2.4623    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   14.1400   -1.7020    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   12.2093   -1.1804    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   11.4490   -1.6173    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   11.8859   -2.3775    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.8089   -0.3078    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.1641   -0.7763    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.3157   -2.4583    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    3.9605   -1.9898    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   18.4986    0.3456    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   18.1117   -0.6562    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
   13.2961   -4.3123    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.1122   -2.5108    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
    2.0000   -1.4428    0.0000 H   0  0  0  0  0  0  0  0  0  0  0  0
  1 23  1  0  0  0  0
  1 28  1  0  0  0  0
  2 19  1  0  0  0  0
  2 40  1  0  0  0  0
  3 25  2  0  0  0  0
  4 30  1  0  0  0  0
  5 30  2  0  0  0  0
  6 31  2  0  0  0  0
  7 17  1  0  0  0  0
  7 39  1  0  0  0  0
  8 38  2  0  0  0  0
  9 44  1  0  0  0  0
  9 73  1  0  0  0  0
 10 44  2  0  0  0  0
 11 23  1  0  0  0  0
 11 25  1  0  0  0  0
 11 26  1  0  0  0  0
 24 12  1  1  0  0  0
 12 31  1  0  0  0  0
 12 50  1  0  0  0  0
 13 14  1  0  0  0  0
 13 29  1  0  0  0  0
 13 32  2  0  0  0  0
 14 33  1  0  0  0  0
 14 36  1  0  0  0  0
 15 34  1  0  0  0  0
 15 38  1  0  0  0  0
 15 57  1  0  0  0  0
 16 33  1  0  0  0  0
 16 58  1  0  0  0  0
 16 59  1  0  0  0  0
 17 35  2  0  0  0  0
 18 37  1  0  0  0  0
 18 40  2  0  0  0  0
 19 37  2  0  0  0  0
 20 38  1  0  0  0  0
 20 43  1  0  0  0  0
 20 60  1  0  0  0  0
 21 40  1  0  0  0  0
 21 71  1  0  0  0  0
 21 72  1  0  0  0  0
 22 45  1  0  0  0  0
 22 74  1  0  0  0  0
 22 75  1  0  0  0  0
 23 24  1  0  0  0  0
 23 46  1  6  0  0  0
 24 25  1  0  0  0  0
 24 47  1  0  0  0  0
 26 27  2  0  0  0  0
 26 30  1  0  0  0  0
 27 28  1  0  0  0  0
 27 29  1  0  0  0  0
 28 48  1  0  0  0  0
 28 49  1  0  0  0  0
 29 51  1  0  0  0  0
 29 52  1  0  0  0  0
 31 35  1  0  0  0  0
 32 34  1  0  0  0  0
 32 53  1  0  0  0  0
 33 34  2  0  0  0  0
 35 37  1  0  0  0  0
 36 54  1  0  0  0  0
 36 55  1  0  0  0  0
 36 56  1  0  0  0  0
 39 41  1  0  0  0  0
 39 42  1  0  0  0  0
 39 44  1  0  0  0  0
 41 61  1  0  0  0  0
 41 62  1  0  0  0  0
 41 63  1  0  0  0  0
 42 64  1  0  0  0  0
 42 65  1  0  0  0  0
 42 66  1  0  0  0  0
 43 45  1  0  0  0  0
 43 67  1  0  0  0  0
 43 68  1  0  0  0  0
 45 69  1  0  0  0  0
 45 70  1  0  0  0  0
M  CHG  2   4  -1  13   1
M  END
"""


@pytest.mark.xfail(strict=True, raises=AssertionError)
def test_CID_166625356(molfile_CID_166625356, run_inchi_exe):
    result = run_inchi_exe(molfile_CID_166625356)
    assert (
        "InChI=1S/C40H36BN7O3S3/c1-5-52-24-12-15-27-30(19-24)37-43-35-28-16-13-25(53-6-2)20-31(28)39-45-36-29-17-14-26(54-7-3)21-32(29)38-44-34(27)46(37)41(47(35)39,48(36)38)51-23-10-8-22(9-11-23)18-33(42)40(49)50-4/h8-17,19-21,33H,5-7,18,42H2,1-4H3/t33-,41-/m0/s1"
        == parse_inchi_from_executable_output(result.output)
    )


@pytest.mark.xfail(strict=True, raises=AssertionError)
def test_CID_6419538(molfile_CID_6419538, run_inchi_exe):
    result = run_inchi_exe(molfile_CID_6419538)
    assert (
        "InChI=1S/C16H17N5O2S/c1-21(20-16(17)24)10-11-8-13(9-14(23-2)15(11)22)19-18-12-6-4-3-5-7-12/h3-10,18-19H,1H2,2H3,(H2-,17,20,24)/b11-10+"
        == parse_inchi_from_executable_output(result.output)
    )


def test_CID_53234134(molfile_CID_53234134, run_inchi_exe):
    result = run_inchi_exe(molfile_CID_53234134)
    inchi = parse_inchi_from_executable_output(result.output)
    re_result = run_inchi_exe(inchi, "-InChI2Struct")
    assert "Conversion failed" not in re_result.stderr
