/*
 * Kylin-video
 *
 * Copyright (C) 2021, Tianjin KYLIN Information Technology Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Liu Cong <liucong1@kylinos.cn>
 *
 */

#include "aboutdialog.h"
#include "ui_aboutdialog.h"
#include <KF5/KWindowSystem/kwindoweffects.h>

#include <QDesktopServices>
#include <QMouseEvent>
#include <QGSettings>
#include <QDebug>

#include "global/global.h"
#include "global/globalsignal.h"
#include "global/xatom-helper.h"

using namespace Global;

AboutDialog::AboutDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::AboutDialog)
{
    ui->setupUi(this);

    setWindowFlags(windowFlags() | Qt::Tool);
    // 字体设置
    QFont f("Noto Sans CJK SC Regular");
    f.setPixelSize(14);
//    ui->label_name->setFont(f);

    ui->label_3->setFont(f);
    ui->label_4->setFont(f);
    ui->textBrowser->setFont(f);

    f.setFamily("Noto Sans CJK SC Medium");
    f.setPixelSize(18);
    ui->label_2->setFont(f);

    connect(ui->btn_close, &QPushButton::clicked, [&](){close();});

    ui->label_3->setText(QString("<html><head/><body><p align=\"center\">%0</p></body></html>").arg(tr("version: ").append("3.1.1")));
    ui->label_4->setText(tr("service and support: ").append("<a href=\"mailto://support@kylinos.cn\">support@kylinos.cn</a>"));
    connect(ui->label_4, SIGNAL(linkActivated(QString)), this, SLOT(slotOpenUrl(QString)));
    initStyle();

    if(isWayland)
        installEventFilter(this);
    else
        setModal(true);
}

AboutDialog::~AboutDialog()
{
    delete ui;
}

void AboutDialog::setBlackTheme()
{
    setStyleSheet("#AboutDialog{background-color:#1F2022}"
                  "#textBrowser{background-color:#1F2022}");
    ui->label_2->setStyleSheet("color:#f0f0f0");
    ui->label_3->setStyleSheet("color:#d9d9d9");
    ui->label_4->setStyleSheet("color:#d9d9d9");
    ui->textBrowser->setStyleSheet("background-color:rgba(0,0,0,0);color:#d9d9d9");
    ui->label_4->setText(tr("Service & Support: ") +
                         "<a href=\"mailto://support@kylinos.cn\""
                         "style=\"color:rgba(225,225,225,1)\">"
                         "support@kylinos.cn</a>");
}

void AboutDialog::setLightTheme()
{
    setStyleSheet("#AboutDialog{background-color:#ffffff}"
                  "#textBrowser{background-color:#ffffff}");
    ui->label_2->setStyleSheet("color:#262626");
    ui->label_3->setStyleSheet("color:#595959");
    ui->label_4->setStyleSheet("color:#595959");
    ui->textBrowser->setStyleSheet("color:#595959");
    ui->label_4->setText(tr("Service & Support: ") +
                         "<a href=\"mailto://support@kylinos.cn\""
                         "style=\"color:rgba(0,0,0,1)\">"
                         "support@kylinos.cn</a>");
}

void AboutDialog::initStyle()
{
    setWindowTitle(tr("Video Player About"));
    setWindowFlags(windowFlags()&~Qt::WindowMaximizeButtonHint);

    ui->btn_icon->setFixedSize(QSize(30,30));
    ui->btn_icon->setIconSize(QSize(30,30));
    ui->btn_icon->setIcon(QIcon::fromTheme("kylin-video"));
    ui->btn_icon->setStyleSheet("QPushButton{border:0px;background:transparent;}"
                                "QPushButton::hover{border:0px;background:transparent;}"
                                "QPushButton::pressed{border:0px;background:transparent;}");

    ui->btn_logo->setFixedSize(QSize(96,96));
    ui->btn_logo->setIconSize(QSize(96,96));
    ui->btn_logo->setIcon(QIcon::fromTheme("kylin-video"));
    ui->btn_logo->setStyleSheet("QPushButton{border:0px;background:transparent;}"
                                "QPushButton::hover{border:0px;background:transparent;}"
                                "QPushButton::pressed{border:0px;background:transparent;}");

    ui->label_title->setText(tr("Video Player"));

    ui->btn_close->setFixedSize(30, 30);
    ui->btn_close->setProperty("isWindowButton", 0x2);
    ui->btn_close->setProperty("useIconHighlightEffect", 0x8);
    ui->btn_close->setFlat(true);
    ui->btn_close->setIcon(QIcon::fromTheme("window-close-symbolic"));

    ui->textBrowser->setFrameShape(QFrame::NoFrame);

    if(!isWayland)
    {
        // 添加窗管协议
        MotifWmHints hints;
        hints.flags = MWM_HINTS_FUNCTIONS | MWM_HINTS_DECORATIONS;
        hints.functions = MWM_FUNC_ALL;
        hints.decorations = MWM_DECOR_BORDER;
        XAtomHelper::getInstance()->setWindowMotifHint(winId(), hints);
    }

    // 根据主题设置样式
    if(g_settings->value("General/follow_system_theme").toBool())
    {
        if(g_gsettings->get("styleName").toString() == STYLE_UKUI_DEFAULT)
            setLightTheme();
        else
            setBlackTheme();
    }
    else
    {
        if(g_settings->value("General/theme").toInt() == 0)
            setLightTheme();
        else
            setBlackTheme();
    }
    connect(g_gsettings, &QGSettings::changed, [&](QString key){
        // 如果不是跟随主题的话直接返回
        if(key == "styleName")
            if(g_settings->value("General/follow_system_theme").toBool())
                if(g_gsettings->get("styleName").toString() == STYLE_UKUI_DEFAULT)
                    setLightTheme();
                else
                    setBlackTheme();
    });
    connect(g_user_signal, &GlobalUserSignal::sigTheme, [&](int theme){
        switch (theme) {
        case 0:
            if(g_gsettings->get("styleName").toString() == STYLE_UKUI_DEFAULT)
                setLightTheme();
            else
                setBlackTheme();
            break;
        case 1:
            setLightTheme();
            break;
        case 2:
            setBlackTheme();
            break;
        default:
            break;
        }
    });
}

void AboutDialog::showEvent(QShowEvent *e)
{
    // 设置模糊
    KWindowEffects::enableBlurBehind(winId(), true);

    QDialog::showEvent(e);
}

bool AboutDialog::eventFilter(QObject *watched, QEvent *event)
{
    if(isWayland)
    {
        if(watched == this)
        {
            if (event->type() == QEvent::MouseButtonPress){
                auto mouseEvent = static_cast<QMouseEvent *>(event);
                if (mouseEvent->buttons() & Qt::LeftButton) {
                    UKUIDecorationManager::getInstance()->moveWindow(this->windowHandle());
                }
            }
        }
    }
    return false;
}

void AboutDialog::slotOpenUrl(QString u)
{
    QUrl url(u);
    QDesktopServices::openUrl(url);
}
